#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
Apply cspell to your code.

"""
import fnmatch
import os
import re
import shutil
import subprocess

import common


# On Windows, we need to be more precise about the executable to be found
# Otherwise false positives will occur
if os.name == 'nt':
    cspell_bin = "cspell.cmd"
else:
    cspell_bin = "cspell"

# ------------------------------------------------------------------------------


# Can we run cspell ?
def check_cspell_install():
    out = shutil.which(cspell_bin)

    return out is None


# ------------------------------------------------------------------------------


# Return True if cspell found errors in specified file
def check_file(files_to_process):

    files_per_line = "\n".join(files_to_process)

    command = [shutil.which(cspell_bin),
               "lint",
               "--no-must-find-files",
               "--no-progress",
               "--file-list", "stdin"]

    p = subprocess.Popen(command,
                         stdout=subprocess.PIPE,
                         stdin=subprocess.PIPE,
                         stderr=subprocess.STDOUT)
    out, err = p.communicate(input=files_per_line.encode())

    if err is not None:
        common.error(f'Cspell failure: {err}')
        return True

    if out is not None:
        out = out.decode()
        for line in out.splitlines():
            common.note(line)

    if p.wait() != 0:
        common.error('Cspell failure')
        return True

    return False


# ------------------------------------------------------------------------------


def cspell(files):
    if check_cspell_install():
        common.error('Failed to launch cspell.')
        common.error('Please follow the installation guide on '
                     + 'https://cspell.org/ to install it.')
        return True

    # Build the magic regex
    patterns = ["*.cpp", "*.cxx", "*.c", "*.hpp", "*.hxx", "*.h", "*.glsl", "*.cu", "*.cuh", "*.py", "*.bat", "*.sh",
                "*.md", "*.xml", "*.yml"]
    file_matcher = re.compile('|'.join(fnmatch.translate(p) for p in patterns))

    repo_root = common.get_repo_root()

    # List the files to process
    files_to_process = [os.path.join(repo_root, file.path) for file in files if file_matcher.match(file.path)]

    # Only do something when needed
    if len(files_to_process) > 0:
        # For now, cspell does not trigger a sheldon error
        return check_file(files_to_process)

    return False


hooks = {
    'cspell': cspell
}
