#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import argparse
import os
import re

import gitlab

parser = argparse.ArgumentParser(
    formatter_class=argparse.RawDescriptionHelpFormatter,
    description='Add a simple comment on a given MR to link the different coverage reports and remove any previous '
                'comment.'
)

parser.add_argument('repository',
                    help='Name of the repository: "sight", "night", or a private repository.')

parser.add_argument('merge_request',
                    metavar='merge-request',
                    help='Identifier of the target merge-request, same as the one present in its URL.')

parser.add_argument('pipeline_id',
                    metavar='pipeline-id',
                    help='Identifier of the pipeline running this script')

parser.add_argument('--verbose',
                    action='store_true',
                    help='Verbose mode')

args = parser.parse_args()

gitlab_token = os.getenv('GITLAB_TOKEN')

if not gitlab_token:
    raise RuntimeError('Environment variable GITLAB_TOKEN is not defined')

gl = gitlab.Gitlab('https://git.ircad.fr',
                   private_token=gitlab_token)

project = None

if args.repository in ["sight", "night"]:
    project = gl.projects.get("sight/" + args.repository)
else:
    projects = gl.projects.list(search=args.repository)
    project = list(filter(lambda x: x.path == args.repository and not hasattr(x, 'forked_from_project'), projects))[0]

try:
    mr = project.mergerequests.get(args.merge_request)
except gitlab.exceptions.GitlabGetError:
    print(f"Unable to found the merge-request !{args.merge_request}")
    exit(1)

if args.verbose:
    print(f'Merge-request #{mr.iid}: {mr.title}')
    print(f'Author: {mr.author["name"]}')
    print(f'Pipeline: #{mr.pipeline["id"]}, status: {mr.pipeline["status"]}, sha: {mr.pipeline["sha"]}')
    print(f'Source branch: {mr.source_branch}')
    print(f'head sha: {mr.diff_refs["head_sha"]}')
    print(f'Target branch: {mr.target_branch}')
    print(f'base sha: {mr.diff_refs["base_sha"]}\n')

mr_base_sha = mr.diff_refs['base_sha']
mr_pipeline_id = mr.pipeline['id']

pipeline = project.pipelines.get(mr_pipeline_id)

coverage_job_re = r'.*linux-debug-gcc' if args.repository == "night" else r'.*coverage-gcc'

for job in pipeline.jobs.list():
    job_match = re.match(coverage_job_re, job.name)
    if job_match and job.stage == 'build':
        pipeline_job = job

if pipeline_job is None:
    print(f'Cannot find a coverage job in pipeline #{mr_pipeline_id}')
    exit(1)

if pipeline_job.status != "success":
    print(f'Last coverage job #{pipeline_job.id} is not yet successful.')
    print(f'Current state: {pipeline_job.status} in pipeline #{mr_pipeline_id}. Cannot proceed')
    exit(1)

target_username = "_bot_"  # Replace with the actual username
target_body = "Coverage report"  # Replace with the actual username

for note in mr.notes.list():
    if target_username in note.author['username'] and target_body in note.body:
        if(args.verbose):
            print("Deleting note " + str(note.id))
        mr.notes.delete(note.id)

repo_path = args.repository + ("/" + args.repository if args.repository == "disrumpere" else "")

mr.notes.create({
    'body': (
        f"**Diff coverage report** is available\n"
        f"[here](https://sight.pages.ircad.fr/-/{repo_path}/-/jobs"
        f"/{args.pipeline_id}/artifacts/.coverage/report.html)\n"
        f"\n"
        f"**Coverage report** is available\n"
        f"[here](https://sight.pages.ircad.fr/-/{repo_path}/-/jobs"
        f"/{pipeline_job.id}/artifacts/.build/coverage/index.html)"
    )
})
